/**
 * Mail Wave Chatbot - Secure Widget Loader
 * Version: 5.0.0 - With User Memory
 * Copyright © 2025 Mail Wave. All rights reserved.
 */

(function() {
    'use strict';
    
    // ═══════════════════════════════════════════════════════
    // SECURITY CONFIGURATION
    // ═══════════════════════════════════════════════════════
    
    const ALLOWED_DOMAINS = [
        'mailwave.dev',
        'mailwave.online',
        'www.mailwave.dev',
        'www.mailwave.online',
        'onecompiler.com', // For testing purposes
        'localhost'
    ];
    
    const WEBHOOK_URL = 'https://mailwave.online/api/chat-proxy.php';
    
    // ═══════════════════════════════════════════════════════
    // DOMAIN VERIFICATION
    // ═══════════════════════════════════════════════════════
    
    function isAllowedDomain() {
        const currentDomain = window.location.hostname;
        if (!currentDomain) return true; // Allow for local file testing
        return ALLOWED_DOMAINS.some(domain => 
            currentDomain === domain || currentDomain.endsWith('.' + domain)
        );
    }
    
    if (!isAllowedDomain()) {
        console.error('Mail Wave Widget: Unauthorized domain - License required. Current domain:', window.location.hostname);
        return;
    }
    
    if (window.mailWaveWidgetLoaded) {
        return;
    }
    
    // ═══════════════════════════════════════════════════════
    // INJECT CSS & HTML
    // ═══════════════════════════════════════════════════════
    
    function injectContent() {
        const style = document.createElement('style');
        style.id = 'mailwave-widget-styles';
        style.textContent = `
* { margin: 0; padding: 0; box-sizing: border-box; } .chat-widget { position: fixed; bottom: 20px; right: 20px; z-index: 10000; font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; } .chat-icon { width: 70px; height: 70px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); border-radius: 50%; display: flex; align-items: center; justify-content: center; cursor: pointer; box-shadow: 0 8px 25px rgba(102, 126, 234, 0.4); transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1); position: relative; overflow: hidden; } .chat-icon:hover { transform: scale(1.1); box-shadow: 0 12px 35px rgba(102, 126, 234, 0.6); } .chat-icon::before { content: ''; position: absolute; top: -50%; left: -50%; width: 200%; height: 200%; background: linear-gradient(45deg, transparent, rgba(255,255,255,0.2), transparent); animation: shimmer 3s infinite; } @keyframes shimmer { 0% { transform: translateX(-100%) translateY(-100%) rotate(45deg); } 100% { transform: translateX(100%) translateY(100%) rotate(45deg); } } .chat-icon svg { width: 35px; height: 35px; color: white; z-index: 1; position: relative; } .chat-icon .close-icon { display: none; } .chat-icon.active .chat-svg { display: none; } .chat-icon.active .close-icon { display: block; } .notification-badge { position: absolute; top: -5px; right: -5px; width: 25px; height: 25px; background: #ef4444; border-radius: 50%; display: flex; align-items: center; justify-content: center; color: white; font-size: 12px; font-weight: bold; border: 3px solid white; animation: pulse-badge 2s infinite; } @keyframes pulse-badge { 0%, 100% { transform: scale(1); } 50% { transform: scale(1.1); } } .chat-container { position: absolute; bottom: 90px; right: 0; width: 380px; height: 600px; background: white; border-radius: 20px; box-shadow: 0 20px 60px rgba(0, 0, 0, 0.15); transform: scale(0) translateY(50px); opacity: 0; visibility: hidden; transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1); overflow: hidden; } .chat-container.active { transform: scale(1) translateY(0); opacity: 1; visibility: visible; } .registration-form { padding: 30px; text-align: center; height: 100%; display: flex; flex-direction: column; justify-content: center; } .registration-form.hidden { display: none; } .form-title { font-size: 24px; font-weight: 600; color: #1f2937; margin-bottom: 10px; } .form-subtitle { color: #6b7280; margin-bottom: 30px; line-height: 1.5; } .form-group { margin-bottom: 20px; text-align: left; } .form-label { display: block; font-weight: 500; color: #374151; margin-bottom: 8px; font-size: 14px; } .form-input { width: 100%; padding: 12px 16px; border: 2px solid #e5e7eb; border-radius: 12px; font-size: 16px; transition: all 0.3s ease; background: #f9fafb; } .form-input:focus { outline: none; border-color: #667eea; background: white; box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1); } .form-input::placeholder { color: #9ca3af; } .submit-btn { width: 100%; padding: 14px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border: none; border-radius: 12px; font-size: 16px; font-weight: 600; cursor: pointer; transition: all 0.3s ease; margin-top: 10px; } .submit-btn:hover { transform: translateY(-2px); box-shadow: 0 8px 25px rgba(102, 126, 234, 0.3); } .submit-btn:active { transform: translateY(0); } .chat-interface { display: none; height: 100%; flex-direction: column; } .chat-interface.active { display: flex; } .chat-header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 20px; position: relative; overflow: hidden; } .chat-header::before { content: ''; position: absolute; top: -50%; left: -50%; width: 200%; height: 200%; background: linear-gradient(45deg, transparent, rgba(255,255,255,0.1), transparent); animation: shimmer 3s infinite; } .header-content { display: flex; align-items: center; position: relative; z-index: 1; } .bot-avatar { width: 50px; height: 50px; background: rgba(255, 255, 255, 0.2); border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-right: 15px; font-size: 24px; } .header-text h3 { font-size: 18px; font-weight: 600; margin-bottom: 5px; } .status { display: flex; align-items: center; font-size: 14px; opacity: 0.9; } .status-dot { width: 8px; height: 8px; background: #10b981; border-radius: 50%; margin-right: 8px; animation: blink 1.5s infinite; } @keyframes blink { 0%, 50% { opacity: 1; } 51%, 100% { opacity: 0.3; } } .chat-messages { flex: 1; overflow-y: auto; padding: 20px; background: #f8fafc; scroll-behavior: smooth; } .chat-messages::-webkit-scrollbar { width: 6px; } .chat-messages::-webkit-scrollbar-track { background: #e2e8f0; border-radius: 3px; } .chat-messages::-webkit-scrollbar-thumb { background: #cbd5e1; border-radius: 3px; } .message { margin-bottom: 15px; animation: messageSlide 0.4s ease-out; } @keyframes messageSlide { from { opacity: 0; transform: translateY(20px); } to { opacity: 1; transform: translateY(0); } } .bot-message { display: flex; align-items: flex-start; } .bot-message .avatar { width: 35px; height: 35px; background: linear-gradient(135deg, #667eea, #764ba2); border-radius: 50%; display: flex; align-items: center; justify-content: center; color: white; font-size: 16px; margin-right: 10px; flex-shrink: 0; } .bot-message .content { background: white; padding: 12px 16px; border-radius: 18px 18px 18px 4px; box-shadow: 0 2px 8px rgba(0,0,0,0.1); max-width: 75%; line-height: 1.6; direction: auto; text-align: start; } .bot-message .content p { margin: 8px 0; direction: auto; text-align: start; } .bot-message .content p:first-child { margin-top: 0; } .bot-message .content p:last-child { margin-bottom: 0; } .bot-message .content a { color: #667eea; text-decoration: none; font-weight: 500; transition: all 0.2s ease; } .bot-message .content a:hover { color: #764ba2; text-decoration: underline; } .bot-message .content strong { font-weight: 600; color: #1f2937; } .bot-message .content ul, .bot-message .content ol { margin: 8px 0; padding-left: 20px; padding-right: 20px; direction: auto; } .bot-message .content li { margin: 4px 0; direction: auto; text-align: start; } .user-message { display: flex; justify-content: flex-end; } .user-message .content { background: linear-gradient(135deg, #667eea, #764ba2); color: white; padding: 12px 16px; border-radius: 18px 18px 4px 18px; max-width: 75%; direction: auto; text-align: start; } .typing-indicator { display: flex; align-items: center; padding: 12px 16px; background: white; border-radius: 18px 18px 18px 4px; box-shadow: 0 2px 8px rgba(0,0,0,0.1); margin-left: 45px; max-width: 75px; } .typing-dots { display: flex; gap: 4px; } .typing-dot { width: 8px; height: 8px; background: #94a3b8; border-radius: 50%; animation: typing 1.4s infinite ease-in-out; } .typing-dot:nth-child(1) { animation-delay: 0s; } .typing-dot:nth-child(2) { animation-delay: 0.2s; } .typing-dot:nth-child(3) { animation-delay: 0.4s; } @keyframes typing { 0%, 60%, 100% { transform: translateY(0); opacity: 0.4; } 30% { transform: translateY(-10px); opacity: 1; } } .quick-replies { display: flex; gap: 8px; margin-top: 10px; flex-wrap: wrap; } .quick-reply { background: rgba(102, 126, 234, 0.1); color: #667eea; border: 1px solid rgba(102, 126, 234, 0.2); padding: 8px 12px; border-radius: 15px; font-size: 13px; cursor: pointer; transition: all 0.3s ease; } .quick-reply:hover { background: #667eea; color: white; transform: translateY(-2px); } .chat-input-container { padding: 20px; background: white; border-top: 1px solid #e2e8f0; } .chat-input-wrapper { display: flex; align-items: center; background: #f1f5f9; border-radius: 25px; padding: 5px; transition: all 0.3s ease; } .chat-input-wrapper:focus-within { background: #e2e8f0; box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1); } .chat-input { flex: 1; border: none; background: transparent; padding: 12px 15px; font-size: 14px; outline: none; color: #334155; direction: auto; } .chat-input::placeholder { color: #94a3b8; } .chat-input:dir(rtl) { text-align: right; } .chat-input:dir(ltr) { text-align: left; } .send-button { width: 40px; height: 40px; background: linear-gradient(135deg, #667eea, #764ba2); border: none; border-radius: 50%; color: white; cursor: pointer; display: flex; align-items: center; justify-content: center; transition: all 0.3s ease; } .send-button:hover { transform: scale(1.05); } .send-button:active { transform: scale(0.95); } @media (max-width: 480px) { .chat-container { width: 320px; height: 550px; bottom: 80px; right: -10px; } .chat-widget { bottom: 15px; right: 15px; } } @media (max-width: 360px) { .chat-container { width: 300px; right: -20px; } }
        `;
        document.head.appendChild(style);
        
        const widget = document.createElement('div');
        widget.className = 'chat-widget';
        widget.innerHTML = `
<div class="chat-icon" id="chatIcon"><div class="notification-badge" id="notificationBadge">1</div><svg class="chat-svg" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"/></svg><svg class="close-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><line x1="18" y1="6" x2="6" y2="18"/><line x1="6" y1="6" x2="18" y2="18"/></svg></div>
<div class="chat-container" id="chatContainer">
<div class="registration-form" id="registrationForm"><div class="form-title">Welcome to Mail Wave</div><div class="form-subtitle">Before we continue, please enter your full name and email address to save your conversation.</div><div class="form-group"><label class="form-label" for="userName">Name</label><input type="text" class="form-input" id="userName" placeholder="John Doe" required></div><div class="form-group"><label class="form-label" for="userEmail">Email</label><input type="email" class="form-input" id="userEmail" placeholder="john@example.com" required></div><button class="submit-btn" id="submitBtn">Start Chat</button></div>
<div class="chat-interface" id="chatInterface"><div class="chat-header"><div class="header-content"><div class="bot-avatar">✓</div><div class="header-text"><h3>Mail Wave Assistant</h3><div class="status"><div class="status-dot"></div>Online</div></div></div></div><div class="chat-messages" id="chatMessages"></div><div class="chat-input-container"><div class="chat-input-wrapper"><input type="text" class="chat-input" placeholder="Type your message here..." id="messageInput"><button class="send-button" id="sendBtn"><svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><line x1="22" y1="2" x2="11" y2="13"></line><polygon points="22,2 15,22 11,13 2,9 22,2"></polygon></svg></button></div></div></div>
</div>
        `;
        document.body.appendChild(widget);
    }
    
    // ═══════════════════════════════════════════════════════
    // WIDGET FUNCTIONALITY
    // ═══════════════════════════════════════════════════════
    
    function initializeWidget() {
        let isOpen = false, isRegistered = false, userData = {}, conversationContext = [];
        const sessionId = 'session_' + Math.random().toString(36).substr(2, 9) + '_' + Date.now();
        const els = { icon: document.getElementById('chatIcon'), container: document.getElementById('chatContainer'), form: document.getElementById('registrationForm'), interface: document.getElementById('chatInterface'), messages: document.getElementById('chatMessages'), input: document.getElementById('messageInput'), badge: document.getElementById('notificationBadge'), submitBtn: document.getElementById('submitBtn'), sendBtn: document.getElementById('sendBtn') };
        const sounds = {send: () => {}, receive: () => {}};
        try { const AudioContext = window.AudioContext || window.webkitAudioContext; const sendCtx = new AudioContext(); sounds.send = () => { const o = sendCtx.createOscillator(), g = sendCtx.createGain(); o.connect(g); g.connect(sendCtx.destination); o.frequency.setValueAtTime(800, sendCtx.currentTime); o.frequency.exponentialRampToValueAtTime(1000, sendCtx.currentTime + 0.1); g.gain.setValueAtTime(0.3, sendCtx.currentTime); g.gain.exponentialRampToValueAtTime(0.01, sendCtx.currentTime + 0.1); o.start(sendCtx.currentTime); o.stop(sendCtx.currentTime + 0.1); }; const recCtx = new AudioContext(); sounds.receive = () => { const o1 = recCtx.createOscillator(), o2 = recCtx.createOscillator(), g = recCtx.createGain(); o1.connect(g); o2.connect(g); g.connect(recCtx.destination); o1.frequency.setValueAtTime(600, recCtx.currentTime); o2.frequency.setValueAtTime(900, recCtx.currentTime); o1.frequency.exponentialRampToValueAtTime(700, recCtx.currentTime + 0.15); o2.frequency.exponentialRampToValueAtTime(1050, recCtx.currentTime + 0.15); g.gain.setValueAtTime(0.2, recCtx.currentTime); g.gain.exponentialRampToValueAtTime(0.01, recCtx.currentTime + 0.15); o1.start(recCtx.currentTime); o2.start(recCtx.currentTime); o1.stop(recCtx.currentTime + 0.15); o2.stop(recCtx.currentTime + 0.15); }; } catch (e) { console.warn("Audio could not be initialized."); }

        // === START: REMEMBER USER LOGIC ===
        function loadUserFromStorage() {
            const savedUserData = localStorage.getItem('mailWaveUserData');
            if (savedUserData) {
                try {
                    userData = JSON.parse(savedUserData);
                    isRegistered = true;
                    els.form.classList.add('hidden');
                    els.interface.classList.add('active');
                    addMsg(`Welcome back, ${userData.name.split(' ')[0]}! 👋 How can I assist you today?`);
                    addQuickReplies();
                } catch (e) {
                    console.error("Could not parse saved user data.", e);
                    localStorage.removeItem('mailWaveUserData'); // Clear corrupted data
                }
            }
        }
        loadUserFromStorage();
        // === END: REMEMBER USER LOGIC ===
        
        els.icon.onclick = () => { isOpen = !isOpen; els.container.classList.toggle('active'); els.icon.classList.toggle('active'); if(isOpen) { els.badge.style.display = 'none'; setTimeout(() => (isRegistered ? els.input : document.getElementById('userName')).focus(), 100); } };
        
        els.submitBtn.onclick = async () => { 
            const name = document.getElementById('userName').value.trim();
            const email = document.getElementById('userEmail').value.trim(); 
            if (!name || !email || !/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email)) { 
                alert('Please enter a valid name and email.'); 
                return; 
            } 
            userData = {name, email}; 
            
            // === ADDITION: Save user data to localStorage ===
            try {
                localStorage.setItem('mailWaveUserData', JSON.stringify(userData));
            } catch (e) {
                console.error("Could not save user data to localStorage.", e);
            }
            // === END ADDITION ===
            
            isRegistered = true; 
            els.form.classList.add('hidden'); 
            els.interface.classList.add('active'); 
            try { await fetch(WEBHOOK_URL, {method: 'POST', headers: {'Content-Type': 'application/json'}, body: JSON.stringify({event: 'registration', sessionId, user: {name, email}})}); } catch(e){ console.error(e); } 
            addMsg(`Thank you, ${name.split(' ')[0]}! I'm your Mail Wave assistant. I can help you with:\n- Creating temporary email addresses\n- Understanding our pricing plans\n- Technical support\n- Account questions\n\nHow can I assist you today?`); 
            addQuickReplies(); 
            els.input.focus(); 
        };
        
        async function sendMsg() { if (!isRegistered) return; const msg = els.input.value.trim(); if (!msg) return; sounds.send(); addUserMsg(msg); els.input.value = ''; const typing = showTyping(); try { const res = await fetch(WEBHOOK_URL, {method: 'POST', headers: {'Content-Type': 'application/json', 'Accept': 'application/json'}, body: JSON.stringify({body: {sessionId, user_data: userData, chatInput: msg, context: conversationContext.slice(-3)}})}); typing.remove(); sounds.receive(); if (res.ok) { const txt = await res.text(); try { const json = JSON.parse(txt); addMsg(json.output || txt); conversationContext.push({user: msg, bot: (json.output || txt)}); } catch { addMsg(txt); conversationContext.push({user: msg, bot: txt}); } } else { addMsg("I'm here to help with Mail Wave!"); } } catch (err) { typing.remove(); sounds.receive(); addMsg("Sorry, I'm having trouble connecting. Please try again later."); console.error("Chatbot fetch error:", err); } }
        
        els.sendBtn.onclick = sendMsg;
        els.input.onkeypress = (e) => { if (e.key === 'Enter' && !e.shiftKey) { e.preventDefault(); isRegistered ? sendMsg() : els.submitBtn.onclick(); } };
        
        function addMsg(txt) { const d = document.createElement('div'); d.className = 'message bot-message'; d.innerHTML = `<div class="avatar">🤖</div><div class="content">${formatMd(txt)}</div>`; els.messages.appendChild(d); els.messages.scrollTop = els.messages.scrollHeight; }
        function addUserMsg(txt) { const d = document.createElement('div'); d.className = 'message user-message'; d.innerHTML = `<div class="content">${txt.replace(/</g, '&lt;').replace(/>/g, '&gt;')}</div>`; els.messages.appendChild(d); els.messages.scrollTop = els.messages.scrollHeight; }
        function showTyping() { const d = document.createElement('div'); d.className = 'message bot-message'; d.innerHTML = `<div class="avatar">🤖</div><div class="typing-indicator"><div class="typing-dots"><div class="typing-dot"></div><div class="typing-dot"></div><div class="typing-dot"></div></div></div>`; els.messages.appendChild(d); els.messages.scrollTop = els.messages.scrollHeight; return d; }
        function addQuickReplies() { const d = document.createElement('div'); d.className = 'quick-replies'; const replies = [{label: 'What is Mail Wave?', value: 'What is Mail Wave?'}, {label: 'Pricing Plans', value: 'Show me pricing plans'}, {label: 'Contact Support', value: 'Contact support team'}]; d.innerHTML = replies.map(r => `<div class="quick-reply">${r.label}</div>`).join(''); d.querySelectorAll('.quick-reply').forEach((el, i) => { el.onclick = () => { document.getElementById('messageInput').value = replies[i].value; sendMsg(); }; }); if (els.messages.lastElementChild) { els.messages.lastElementChild.querySelector('.content').appendChild(d); } }
        function formatMd(t) { if (typeof t !== 'string') return ''; let h = t.replace(/</g, '&lt;').replace(/>/g, '&gt;').replace(/\[([^\]]+)\]\(([^\)]+)\)/g, '<a href="$2" target="_blank" rel="noopener noreferrer">$1</a>').replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>').replace(/\*(.*?)\*/g, '<em>$1</em>'); const ls = h.split('\n'); let r = [], inL = false, lType = ''; for (let i = 0; i < ls.length; i++) { const l = ls[i]; if (l.trim().startsWith('- ')) { if (!inL || lType !== 'ul') { if (inL) r.push(`</${lType}>`); r.push('<ul>'); inL = true; lType = 'ul'; } r.push('<li>' + l.trim().substring(2) + '</li>'); } else { if (inL) { r.push(`</${lType}>`); inL = false; lType = ''; } if (l.trim()) r.push('<p>' + l + '</p>'); } } if (inL) r.push(`</${lType}>`); return r.join(''); }
    }
    
    // ═══════════════════════════════════════════════════════
    // INITIALIZATION
    // ═══════════════════════════════════════════════════════
    
    function init() {
        injectContent();
        initializeWidget();
        window.mailWaveWidgetLoaded = true;
    }
    
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
    
})();